package Quancom::Test::Server;

# This package is for internal use only - for testing the
# Quancom.pm module and should work like the real USB-OPTO device of
# Quancom.

use strict;
use warnings;
use Carp;
use IO::Socket;
use IO::Select;

my $STX = "\x02";

sub new {
    my $class = ref $_ ? ref shift : shift;

    my $self = bless {} => $class;
    my $addr = shift or croak "need socket address";

    $self->{debug} = 0;

    # if there's a parameter passed we understand it as
    # a socket address for communication
    if ($addr eq "-") {
        warn "listening on: stdio\n";
    }
    else {
        if ($addr =~ /\//) {
            $self->{file}   = $addr;
            $self->{socket} = new IO::Socket::UNIX(
                Listen => 1,
                Local  => $self->{file}
            ) or croak "Can't create IO::Socket::UNIX: $!\n";
        }
        else {
            $addr = "127.0.0.1:$1" if $addr =~ /^:?(\d+)/;

            $self->{socket} = new IO::Socket::INET(
                Listen    => 1,
                ReuseAddr => 1,
                LocalAddr => $addr
            ) or croak "Can't create IO::Socket::INET: $!\n";
        }

        warn "listening on: $addr\n";
    }

    # we can't use 64bit as Vector (vec()), since not all platforms support it
    # with this length

    @{ $self->{outputs} } = map { pack "c", $_ } (0, 0, 0, 0, 0, 0, 0, 0);
    $self->show;

    return $self;
}

sub show {
    my $self = shift;
    printf STDERR "%0v8b\n", join "", @{ $self->{outputs} };
}

sub DESTROY {
    my $self = shift;
    unlink $self->{file} if $self->{file};
}

sub run {
    my $self = shift;

    if ($self->{socket}) {

        # It's a quick and dirty runner!
        # This runner lives with the assumption, that the client always
        # sends a line terminated by "\r" in one chunk. No other processing
        # takes place between the first character and the final "\r",
        # especially no accepting of new connections or reading of other
        # connection data or sending data!
        # BUT: This socket based server can talk to more than one
        # client.

        my $s = new IO::Select $self->{socket};
        while (my @ready = $s->can_read) {
            foreach my $c (@ready) {

                # create a new connection or process incoming
                # data

                if ($c == $self->{socket}) {
                    my $n = $self->{socket}->accept;
                    $n->autoflush(1);
                    $s->add($n);
                    next;
                }

                local $/ = "\r";    # quancom sends CR as line terminator
                my $l = <$c>;
                $s->remove($c), next if not defined $l;
                chomp $l;
                $l = $self->_process($l);
                $c->print($l . "\r");
            }
        }
        return;
    }

    # STDIO communication
    while (<>) {
        chomp;
        print $self->_process($_), "\n";
    }
}

sub _process {
    my $self = shift;
    my $line = shift;
    my $retval;

    # some fixups
    $line =~ s/^$STX//;    # cut STX, if any
    $line =~ s/(..)$//;    # cut checksum

    warn "got: <STX>$line($1)\n" if $self->{debug};

    return pack("ac", "E", 1)    # checksum error
      if $1 ne ".." and hex($1) != unpack("%8C*", "$STX$line");

    my ($jid, $cmd, $width, $addr, $data) = (
        $line =~ /
	([\da-f]{2})	# jid
	((?-i)[RWSC])	# cmd
	((?-i)[BWLX])	# width
	([\da-f]{4})	# addr
	(.*?)		# data
	$/xi
    );

    # some transformations for more easy use
    $addr = hex($addr);
    $width =
        $width eq "B" ? 1
      : $width eq "W" ? 2
      : $width eq "L" ? 3
      : $width eq "X" ? 4
      :                 0;

    if ($cmd eq "W") {
        my @data = $data =~ /(..)/g;
        return pack("ac", "E", 3)    # width error
          if @data != $width;

        my $offset = $addr - 0x100;

        @{ $self->{outputs} }[ $offset .. $offset + $width - 1 ] =
          map { pack "C", hex($_) } @data;
        $self->show;

        $retval = "O$jid";
    }
    else {
        warn "command \"$cmd\" not supported\n";
        $retval = pack("ac", "E", 2);
    }

    return $retval . sprintf("%02x", unpack("%8C*", $retval));
}

1;
