package Quancom;

#    Quancom - a perl library module to talk to USBOPTO-XX
#    Copyright (C) 2008  Heiko Schlittermann
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#    Heiko Schlittermann <hs@schlittermann.de>

use strict;
use warnings;
use Carp;
use IO::Socket::INET;    # FIXME: shold be loaded conditionally
use IO::Socket::UNIX;    # FIXME: shold be loaded conditionally

use Quancom::Result;

our $VERSION = 0.1;

my $DEFAULT_PORT = 1001;

sub new {
    my $class = ref $_[0] ? ref shift : shift;
    my $self = bless {} => $class;

    $self->{peer} = shift or croak "need a peer address!";

    if ($self->{peer} !~ /\//) {
        $self->{peer} .= ":$DEFAULT_PORT"
          unless $self->{peer} =~ /:\d+$/;

        $self->{socket} = new IO::Socket::INET(
            Proto    => "tcp",
            PeerAddr => $self->{peer}
        );
    }
    else {
        $self->{socket} = new IO::Socket::UNIX(Peer => $self->{peer});
    }

    $self->{socket} or croak "Can't create socket to $self->{peer}: $!\n";

    $self->{job} = 0;
    $self->{ok}  = undef;

    return $self;
}

sub last_result { $_[0]->{last_result} }

sub cmd {
    my $self = shift;
    my $cmd  = shift;

    $self->_tx($cmd);
    $self->_rx($cmd);

    return $self->{last_result};
}

sub TIESCALAR {
    my $class = shift;
    my ($ip)  = @_;
    my $self  = bless {} => $class;
    warn "tied to ip $ip\n";

    return $self;
}

sub STORE {
    my $self = shift;
    my ($key, $value) = @_;

    #croak "invalid value \"$value\" (should be 0 or 1)\n";
    warn "Set $key to $value\n";
}

sub _tx {
    my $self = shift;
    my $cmd  = shift;

    $self->{job} = ++$self->{job} % 255;    # cap the job id on 255;
    $cmd = "\x02" . sprintf("%02x", $self->{job}) . $cmd;   # add STX and job id
    $cmd .= sprintf("%02x", unpack("%8C*", $cmd));          # add checksum

    warn "sending $cmd\n";
    $self->{socket}->print($cmd . "\r");
}

sub _rx {
    my $self = shift;

    local $/ = "\r";    # CR is the delimiter
    $self->{last_result} = new Quancom::Result($self->{socket}->getline);
}

1;

__END__

=head1 NAME

Quancom - perl module to access the usb opto quancom device

=head1 SYNOPSIS

    use Quancom;

    my $quancom = new Quancom "172.16.0.22";
    my $result  = $q->cmd("xxxxxx");
    if ($result->error) { die $result->error_message } 
    else { print $result->data }


=head1 METHODS

=over

=item constructor B<new>( I<ip or socket name> )

This method returns a new Quancom object if the connection was
successfully established. For testing you may use "0.0.0.0" as address,
this disables the socket communication and just simulates the Quancom
module.

=item B<cmd>( I<string> )

Send a Quancom string to the device. The string here should be
B<without> the leading STX and Jobid as well without the trailing CR.
It returns a L<Quancom::Result> object.

The only tested I<string> is currently "WB0101FF", which should set
all bits on the first relais. Some other (untested) string for setting
just the lowest bit on the first relais should be "WB010001".

=item B<last_result>( )

This returns an object containing the last result.
See L<Quancom::Result> for more information.

=back

=head1 MORE EXAMPLES

    use Quancom;
    my $quancom = new Quancom("172.20.20.1");
    die "Sorry" if $quancom->cmd("xxxx")->error;

=head1 SEE ALSO

L<Quancom::Result>

=head1 AUTHOR

    Maik Schueller
    Heiko Schlittermann

=head1 THANKS

Thanks to Mr. van den Berg from Quancom for his support.

=cut
