package Quancom::Result;

use strict;
use warnings;

sub new {
    my $class = ref $_[0] ? ref shift : shift;
    return bless {} => $class;
}

sub ok {
    my $self = shift;
    return $self->{ok};
}

sub result {
    my $self = shift;
    return undef if not $self->{ok};
    return $self->{result};
}

sub error {
    my $self = shift;
    return undef if $self->{ok};
    return $self->{error_code};
}

sub error_message {
    my $self = shift;

    return undef if !@_ and $self->{ok};

    return ("checksum error", "character error", "invalid command",
        "invalid width")[ @_ ? $_[0] : $self->{error_code} ];
}

1;

__END__

=head1 NAME

Quancom::Result - perl module to access the usb opto quancom device result

=head1 SYNOPSIS

    use Quancom;

    my $q = new Quancom 172.16.0.22;
    my $r = $q->cmd("xxxxxx") 
	or die $r->error_message;

=head1 METHODS

=over

=item B<ok> ( )

Use this method to query the last operations status.

=item B<result> ( )

Returns the last result. This is valid only if the last status is ok,
otherwise you'll get "undef".

=item B<error_message> ( [I<error code>] )

Returns a message describing the last error. Of if you pass an error
code it will the return the associated message.

=item B<error> ( )

Returns the last error code (numerical).

=back

=head1 AUTHOR

    Maik Schueller
    Heiko Schlittermann

=cut
