/*
 * Source File : me8100_test_int.c                                            
 * Destination : me8100_test_int.out                                              
 * Author      : GG (Guenter Gebhardt)                                 
 *    
 *                                                                     
 * File History: Version   Date       Editor   Action                  
 *---------------------------------------------------------------------
 *               1.00.00   01.07.12   GG       first release           
 *                                                                     
 *---------------------------------------------------------------------
 *                                                                     
 * Description:
 *   This program shows the usage of the driver and the interrupt
 *   facility of the me8100. First the board is configured, in order to
 *   generate an interrupt when a bit pattern of 0x0001 on port a and a bit
 *   pattern of 0x0100 an port b is pending. Then the board is configured, 
 *   in order to generate an interrupt with a bit mask of 0x0001 on port a 
 *   and a bit mask of 0x0100 on port b. 
 *   We install a signal handler, which is informed by the interrupt routine 
 *   by signalling of the driver, when a interrupt occures.
 */
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <stdlib.h>
#include "me8100.h"


/* Prototypes */
static void signal_handler(int);

/* Counts the interrupts */
static me8100_int_occur_type intcounts;

/* Count of signal handler execution */
static int i = 0;

/* Path to the ME8100 board */
static int file_handle = -1;

int main(void){
  int err = 0;
  int oflags = 0;

  unsigned short mask_a;
  unsigned short ctrl_a;

  unsigned char icsr;

  printf("IRQ Test %d\n", getpid());

  file_handle = open("/dev/me8100_0", O_RDWR, 0);

  if(file_handle < 0){
    printf("Cannot open path !\n");
    return 1;
  }

  /*---------------------- general setup ------------------------------------*/

  /* install the signal handler */
  signal(SIGIO, signal_handler);

  /* set current process as owner of the path */
  fcntl(file_handle, F_SETOWN, getpid());

  /* read the flags of the path */
  oflags = fcntl(file_handle, F_GETFL);

  /* Inform the driver to put the current process on the fasync queue */
  fcntl(file_handle, F_SETFL, oflags | FASYNC); 

  /* enable both interrupts on the plx, set interrupts to high active */
  icsr =
    LOCAL_INT1_EN |
    LOCAL_INT1_POL |
    PCI_INT_EN;

  err = ioctl(file_handle, ME8100_SETUP_ICSR, &icsr);
  if(err){
    printf("Cannot setup PLX\n");
    return 1;
  }

  /*-------------------- Interrupt caused by bit mask -----------------*/

  /* Set the proper bit mask for port a */
  mask_a = 0xffff;
  err = ioctl(file_handle, ME8100_WRITE_MASK_A, &mask_a);
  if(err){
    printf("Cannot write mask a\n");
    return 1;
  }

  /* Enable interrupt signalling by bit mask for port a */
  ctrl_a = ME8100_CTL_ENIO | ME8100_CTL_SOURCE | ME8100_CTL_IRQ_MASK;
  err = ioctl(file_handle, ME8100_WRITE_CTRL_A, &ctrl_a);
  if(err){
    printf("Cannot write ctrl a\n");
    return 1;
  }

  printf("<<<--- WAITING FOR INTERRUPTS BY BIT MASK --->>>\n\n");

  i = 0;
  while(i < 10) {
    select(0, NULL, NULL, NULL, NULL);
  }

  printf("Close path to me8100_0\n");
  err = close(file_handle);
  if(err){
    printf("Kann Pfad nicht schliessen\n");
    return 1;
  }

  return 1;
}



void signal_handler(int sig){
  int err = 0;
  i++;
  err = ioctl(file_handle, ME8100_GET_INT_COUNT, &intcounts);
  if(err)
    return;

  fprintf(stderr, "<<<--- ME8100 SIGNAL HANDLER CALLED --->>>\n"
	 "Execution   = %04d\n"
	 "int_count_1 = %04d\n"
	 "int_count_2 = %04d\n\n", i, intcounts.int1, intcounts.int2);
  return;
}

/*
	vim:sts=2 sw=2 aw ai sm:
*/
