package ME8100;
# (c) 2002 Heiko Schlittermann
# For POD documentation see the end of this file.

use strict;
use Fcntl;
use IO::File;
use IO::Select;
use Carp;

my %Objects = ();
my $gotSignal = 0;
my $opt_async = 0;

sub _iohandler($);
sub import { $opt_async = grep { /^:async/ } @_; }

# Install the signal handler only if we have passed the ':async' tag
# to the module...
INIT { $SIG{IO} = \&_iohandler if $opt_async; }

# Create a new object. Open all the named devices (read only)
# and read the current values (as the driver guarantees the
# very first read to succeed).
# The order depends on the order the device names are passed 
# to the new() method;
sub new {
    my $self = {};
    my $class = shift;
    bless $self, $class;

    my (@fds, @inputs);
    my %idx = ();
    my $idx = 0;
    my $inputBits = "";

    foreach my $device (@_) {
	my ($flags, $val);
	my $fd = new IO::File($device, "r")
	    or croak("open($device): $!\n");

	if ($opt_async) {
	    fcntl($fd, F_SETOWN, $$) or croak "Can't set owner: $!\n";
	    $flags = fcntl($fd, F_GETFL, 0) or croak "Can't get flags: $!\n";
	    fcntl($fd, F_SETFL, $flags | O_ASYNC) or croak "Can't set flags: $!\n";
	}

	$Objects{$fd} = { object => $self, fd => $fd };
	$idx{$fd} = $idx;
	vec($inputBits, $idx, 16) = 0x0;
	push @fds, $fd;

	++$idx;
    }

    $self->{idx} = {%idx};	    # fd -> index in input word array
    $self->{fds} = [@fds];	    # list of fds
    $self->{select} = new IO::Select(@fds);

    $self->{bits} = $idx * 16;
    $self->{inputBits} = $inputBits;
    $self->{oldBits} = undef;		
    $self->{changedBits} = "";

    $self->{changed} = [split //, "0" x $self->{bits} ];
    return $self;
}

sub read {
    # This functions should read a set of values from the board.  But: a
    # race condition might occur: while we're waiting for the select()
    # to complete, the SIGIO might be catched.  Both, SIGIO as well as
    # the suddenly succeeding select() indicate a possible successful
    # read... But only one of them will be successful!

    my ($self, %args) = @_;

    {
	local $SIG{IO} = sub { $gotSignal = $_[0] } if $opt_async;
	my @ready = $self->{select}->can_read($args{timeout});

	if (!@ready) {
	    warn "select() returned nothing: $!\n";
	    return undef;
	}

	$self->_read(@ready);
    }

    $gotSignal and _iohandler($gotSignal);
    return split //, unpack("b*", $self->{inputBits});
}

sub status {
    my $self = shift;
    return split //, unpack("b*", $self->{inputBits});
}

# Read *really* from the board and store the result at the proper
# element of our @inputs array.
sub _read($@) {
    my ($self, @fds) = @_;
    my $val;	# bit0-7 bit8-15

    foreach my $fd (@fds) {
	my $idx = $self->{idx}->{$fd};
	$self->{fds}->[$idx]->sysread($val, 2) or carp("sysread(): $!\n");
	vec($self->{inputBits}, $idx, 16) = unpack("n2", $val);
    }

    # Now get the difference between the old bits and the current values...
    # and then add these values to the array containing the change counters
    
    $self->{oldBits} = $self->{inputBits} if not defined $self->{oldBits};
    my $changedBits = $self->{inputBits} ^ $self->{oldBits};

    my @changed = split //, unpack("b*", $changedBits);
    #print STDERR "o: ", unpack("b*", $self->{oldBits}), "\n";
    #print STDERR "n: ", unpack("b*", $self->{inputBits}), "\n";
    #print STDERR "=: ", @changed, "\n\n";

    for (my $i = 0; $i < $#changed; ++$i) {
	$self->{changed}->[$i] += $changed[$i];
    }

    $self->{changedBits} |= $changedBits;
    $self->{oldBits} = $self->{inputBits};
}

sub _iohandler($) { 
    # If an interrupt occurs, we've to search for the file descriptor(s)
    # that caused the interrupt.  This is done by a lookup in the module global
    # %Objects hash.  ("fd" -> { fd => XXX, object => XXXX })

    my $signal = shift;

    # I do some magic(?) here, since the key isn't usable a a refence
    # anymore, we stored the reference itself behind the key, as well
    # as the object reference...
    
    my $select = new IO::Select(map { $Objects{$_}->{fd} } keys %Objects);
    my @ready = $select->can_read(0);

    foreach (@ready) {
	my $object = $Objects{$_}->{object};
	my $fd = $Objects{$_}->{fd};
	$object->_read($fd);
    }
    $gotSignal = 0;
};


sub changed {
    my $self = shift;
    my @changed = @{$self->{changed}};
    my $r = 0;

    $self->{changedBits} = "";
    $self->{changed} = [ split //, "0" x $self->{bits} ];

    wantarray and return @changed;

    foreach (@changed) {
	$r += $_;
    }
    return $r;
}

#sub DESTROY
#{
#    my $self = shift;
#    delete @Objects{ @{$self->{fds} }};
#}


#-- Documenation follows 

=head1 NAME

    ME8100 - a perl interface to the me8100 driver (Meilhaus Digital I/O)

=head1 SYNOPSIS

    use ME8100;
or
    use ME8100 qw/:async/;

    $me8100 = new ME8100 qw(/dev/me8100_0a /dev/me8100_0b);
    @bits = $me8100->read();
    @bits = $me8100->status();

=head1 DESCRIPTION

This module is an interface to the me8100 driver talking with the Meilhaus
D I/O board ME8100.

=over 4

=item new(I<device>,...)

Creates a new ME8100 object connected to all the listed devices.

=item read(I<[timeout =E<gt> timeout]>)

Read the input from the devices.  This call is guaranteed to succeede
on the first run, but every following call will block until the status
of the input changed.  (This behaviour is due to the driver.)

A timeout may be passed.

On success an array of the input bits is returned, otherwise undef.

=item status()

Returns an array with the current status of all input bits.
Never blocks.  No side effects.

=item changed()

In array context returns an array containing the counts of changes
for every single bit.  In scalar context returns the total number of
changes.

Every call resets the counter.

=item _iohandler(I<signal>)

For internal use only.

=item _read()

For internal use only.

=item DESTROY()

For internal use only.

=head1 AUTHOR

Heiko Schlittermann <hs@schlittermann.de>

=head1 COPYRIGHT

The GNU Copyright applies.

=cut

1;
# vim:sts=4 sw=4 aw ai sm:
