/*
 * Source File : me8100_test_int.c                                            
 * Destination : me8100_test_int.out                                              
 * Author      : GG (Guenter Gebhardt)                                 
 *    
 *                                                                     
 * File History: Version   Date       Editor   Action                  
 *---------------------------------------------------------------------
 *               1.00.00   01.07.12   GG       first release           
 *                                                                     
 *---------------------------------------------------------------------
 *                                                                     
 * Description:
 *   This program shows the usage of the driver and the interrupt
 *   facility of the me8100. First the board is configured, in order to
 *   generate an interrupt when a bit pattern of 0x0001 on port a and a bit
 *   pattern of 0x0100 an port b is pending. Then the board is configured, 
 *   in order to generate an interrupt with a bit mask of 0x0001 on port a 
 *   and a bit mask of 0x0100 on port b. 
 *   We install a signal handler, which is informed by the interrupt routine 
 *   by signalling of the driver, when a interrupt occures.
 */
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <linux/spinlock.h>

#include "me8100.h"

/* Prototypes */
static void signal_handler(int);

/* Counts the interrupts */
static me8100_int_occur_type intcounts;

/* Count of signal handler execution */
static int i = 0;

/* Path to the ME8100 board */
static int file_handle = -1;

int main(void){
  int err = 0;
  int minor = 0;
  int oflags = 0;

  unsigned short pattern_a;
  unsigned short pattern_b;
  unsigned short mask_a;
  unsigned short mask_b;
  unsigned short ctrl_a;
  unsigned short ctrl_b;

  unsigned char icsr;

  printf("%c%3s", 27, "[2J");
  printf("<<<--- ME8100 TESTPROGRAM FOR INT --->>>\n\n");

  /*
   * You can select up to four me8100 baords, if installed.
   * 0 is the first board.
   */
  printf("Please type in the minor device number of the board to open : ");
  scanf("%d", &minor);
  printf("Open path /dev/me8100_%d !\n\n", minor);
  
  switch(minor){
  case 0:
    file_handle = open("/dev/me8100_0", O_RDWR, 0);
    break;
  case 1:
    file_handle = open("/dev/me8100_1", O_RDWR, 0);
    break;
  case 2:
    file_handle = open("/dev/me8100_2", O_RDWR, 0);
    break;
  case 3:
    file_handle = open("/dev/me8100_3", O_RDWR, 0);
    break;
  default:
    printf("Invalid input !\n");
    return 1;
  }

  if(file_handle < 0){
    printf("Cannot open path !\n");
    return 1;
  }

  /*---------------------- general setup ------------------------------------*/

  /* install the signal handler */
  signal(SIGIO, signal_handler);

  /* set current process as owner of the path */
  fcntl(file_handle, F_SETOWN, getpid());

  /* read the flags of the path */
  oflags = fcntl(file_handle, F_GETFL);

  /* Inform the driver to put the current process on the fasync queue */
  fcntl(file_handle, F_SETFL, oflags | FASYNC); 

  /* enable both interrupts on the plx, set interrupts to high active */
  icsr =
    LOCAL_INT1_EN |
    LOCAL_INT1_POL |
    LOCAL_INT2_EN |
    LOCAL_INT2_POL |
    PCI_INT_EN;

  err = ioctl(file_handle, ME8100_SETUP_ICSR, &icsr);
  if(err){
    printf("Cannot setup PLX\n");
    return 1;
  }



  /*-------------------- Interrupt caused by bit pattern -----------------*/

  /* Set the proper bit pattern for port a */
  pattern_a = 0x1;
  err = ioctl(file_handle, ME8100_WRITE_PATTERN_A, &pattern_a);
  if(err){
    printf("Cannot write pattern a\n");
    return 1;
  }

  /* Set the proper bit pattern for port b */
  pattern_b = 0x100;
  err = ioctl(file_handle, ME8100_WRITE_PATTERN_B, &pattern_b);
  if(err){
    printf("Cannot write pattern b\n");
    return 1;
  }

  /* Enable interrupt signalling by bit pattern for port a */
  ctrl_a = 0x40;
  err = ioctl(file_handle, ME8100_WRITE_CTRL_A, &ctrl_a);
  if(err){
    printf("Cannot write ctrl a\n");
    return 1;
  }

  /* Enable interrupt signalling by bit pattern for port b */
  ctrl_b = 0x40;
  err = ioctl(file_handle, ME8100_WRITE_CTRL_B, &ctrl_b);
  if(err){
    printf("Cannot write ctrl b\n");
    return 1;
  }

  printf("<<<--- WAITING FOR INTERRUPTS BY BIT PATTERN --->>>\n\n");

  i = 0;
  /* execute until 0x8 interrupt will be occured */
  while(i < 0x4){
  }


  /*-------------------- Interrupt caused by bit mask -----------------*/

  /* Set the proper bit mask for port a */
  mask_a = 0x1;
  err = ioctl(file_handle, ME8100_WRITE_MASK_A, &mask_a);
  if(err){
    printf("Cannot write mask a\n");
    return 1;
  }

  /* Set the proper bit mask for port b */
  mask_b = 0x100;
  err = ioctl(file_handle, ME8100_WRITE_MASK_B, &mask_b);
  if(err){
    printf("Cannot write mask b\n");
    return 1;
  }

  /* Enable interrupt signalling by bit mask for port a */
  ctrl_a = 0x60;
  err = ioctl(file_handle, ME8100_WRITE_CTRL_A, &ctrl_a);
  if(err){
    printf("Cannot write ctrl a\n");
    return 1;
  }

  /* Enable interrupt signalling by bit mask for port b */
  ctrl_b = 0x60;
  err = ioctl(file_handle, ME8100_WRITE_CTRL_B, &ctrl_b);
  if(err){
    printf("Cannot write ctrl b\n");
    return 1;
  }

  printf("<<<--- WAITING FOR INTERRUPTS BY BIT MASK --->>>\n\n");

  i = 0;
  /* execute until 0x8 interrupt will be occured */
  while(i < 0x4){
  }


  /*-------------------------------- END ------------------------------------*/

  printf("Close path to me8100_%d\n", minor);
  err = close(file_handle);
  if(err){
    printf("Kann Pfad nicht schliessen\n");
    return 1;
  }

  return 1;
}



void signal_handler(int sig){
  int err = 0;
  i++;
  err = ioctl(file_handle, ME8100_GET_INT_COUNT, &intcounts);
  if(err)
    return;

  printf("<<<--- ME8100 SIGNAL HANDLER CALLED --->>>\n"
	 "Execution   = %04d\n"
	 "int_count_1 = %04d\n"
	 "int_count_2 = %04d\n\n", i, intcounts.int1, intcounts.int2);
  return;
}

/*
 * vim:sts=2 sw=2 aw ai sm:
 */
