#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#include <signal.h>
#include <stdlib.h>
#include "me8100.h"

const char* DEVICE = "/dev/me8100_0a";
const int MAX_FDS = 2;

/* Prototypes */
static void signal_handler(int);

/* Counts the interrupts */
static me8100_int_occur_type intcounts;

/* Count of signal handler execution */
static int i = 0;

/* Path to the ME8100 board */
static int file_handle = -1;

int main(void){
  int oflags;
  int err;

  struct pollfd fds[MAX_FDS];

  printf("IRQ Test %d\n", getpid());

  file_handle = open(DEVICE, O_RDONLY, 0);

  if(file_handle < 0){
    fprintf(stderr, "Can't open path %s: %m\n", DEVICE);
    exit(EXIT_FAILURE);
  }

  /*---------------------- general setup ------------------------------------*/

  /* install the signal handler */
  signal(SIGIO, signal_handler);

  /* set current process as owner of the path */
  err = fcntl(file_handle, F_SETOWN, getpid());
  if (err < 0) {
    perror("fcntl SETOWN");
    exit(EXIT_FAILURE);
  }

  /* read the flags of the path */
  oflags = fcntl(file_handle, F_GETFL);
  if (err < 0) {
    perror("fcntl GETFL");
    exit(EXIT_FAILURE);
  }

  /* Inform the driver to put the current process on the fasync queue */
  err = fcntl(file_handle, F_SETFL, oflags | FASYNC); 
  if (err < 0) {
    perror("fcntl SETFL");
    exit(EXIT_FAILURE);
  }

  fds[0].fd = file_handle;
  fds[0].events = POLLIN;

  printf("Waiting for Interrupts\n\n");

  for(;;sleep(2)) {
    int ready;
    int i;
    unsigned short val;

    fds[0].revents = 0;
    printf("Polling on fd %d\n", file_handle);
    ready = poll(fds, 1, 1000);
    if (ready < 0) { perror("poll()"); exit(EXIT_FAILURE); }

    printf("polled, %d fds are ready\n", ready);

    for (i = 0; i < ready; ++i) {
      printf("fd %d: 0x%0x\n", i, fds[i].revents);
      if (fds[i].revents & POLLIN) {
	read(file_handle, &val, sizeof(val));
	printf("read: 0x%04x\n", val);
      }
    }

    printf("Waiting...\n");
  }

  printf("Close path to %s\n", DEVICE);
  err = close(file_handle);
  if(err){
    fprintf(stderr, "Can't close %s: %m\n", DEVICE);
    exit(EXIT_FAILURE);
  }

  exit(EXIT_SUCCESS);
}



void signal_handler(int sig){
  int err = 0;
  i++;
  err = ioctl(file_handle, ME8100_GET_INT_COUNT, &intcounts);
  if(err)
    return;

  fprintf(stderr, "<<<--- ME8100 SIGNAL HANDLER CALLED --->>>\n"
	 "Execution   = %04d\n"
	 "int_count_1 = %04d\n"
	 "int_count_2 = %04d\n\n", i, intcounts.int1, intcounts.int2);
  return;
}

/*
	vim:sts=2 sw=2 aw ai sm:
*/
