package BlockDev;

use 5.010;
use Moose;
use POSIX;
use autodie qw(:all);
use Cwd qw(abs_path);

our $VERSION = "0.0";

# extend the PATH with *sbin* if not there already
foreach my $p (qw(/sbin /usr/sbin /usr/local/sbin)) {
    $p ~~ [split(/:/ => $ENV{PATH})]
	or $ENV{PATH} = "$p:$ENV{PATH}";
}

has dev => (is => "ro", isa => "Str");

sub list {
    open(my $fh => "/proc/partitions");
    return map { (split)[3] } grep /^\s*\d+/ => <$fh>;
}

# process the arguments before actually calling the
# constructor

sub BUILDARGS {
    my $class = shift;
    my ($key, $value) = @_;

    given ($key) {
        when ("dev") {
	    $value = "/dev/$value" if $value !~ /^\// and not -e $value and -b "/dev/$value";
            -b $value or die "no blockdevice `$value'";
            return { dev => $value };
        }
        when ("dir") {
            -d $value or die "`$value' is not a directory";
            `df --portability '$value'` =~ /^(?<dev>\/\S+).*?(?<mp>\S+)$/m;
            die "$value does not have a real file system"
              if not $+{dev};
            return { dev => $+{dev} };
        }
        when (/^m(?:ount)?p(?:oint)?$/) {
            -d $value or die "`$value` is not a directory";
            open(my $mounts => "/proc/mounts");
            join("" => <$mounts>) =~ /^(?<dev>\S+)\s$value\s/m
              or die "$value is not a mountpoint";
            return { dev => $+{dev} };
        }
	when ("uuid") {
	    chomp(my $dev = `blkid -c /dev/null -U '$value'`);
	    die "'$value' does not exist as UUID" if not $dev;
	    return { dev => $dev }
	}
	when ("label") {
	    chomp(my $dev = `blkid -c /dev/null -L '$value'`);
	    die "'$value' does not exist as label" if not $dev;
	    return { dev => $dev }
	}
        default { die "unknown initialization" }
    }
}

sub path  { abs_path shift->dev }
sub major { (stat shift->dev)[6] >> 8 }
sub minor { (stat shift->dev)[6] & 0xff }
sub size  {
    my $self = shift;
    open(my $fh => $self->path);
    seek($fh, 0, SEEK_END);
    return tell($fh);
}


1;
__END__

sub new : Validate(uuid => 0, dev => 0) : method {
    my $class = shift;
    my $self = bless \my $x => $class;
    my %arg = @_;
    $X{$self} = \%arg;

    if ($arg{dev}) {
	die "ERROR device `$arg{dev}': $!\n"
	    if not -b $arg{dev};
	$self->_rdev_by_dev;
    }

    return $self;
}

sub major { return $X{+shift}{major} }
sub minor { return $X{+shift}{minor} }

sub _rdev_by_dev {
    my $self = shift;
    $X{$self}{major} = (stat $X{$self}{dev})[6] >> 8;
    $X{$self}{minor} = (stat _)[6] & 0xFF;
}

sub rdev {
    my $self = shift;
    return $self->major, $self->minor if wantarray;
    return join " " => $self->major, $self->minor;
}

sub dev {
    my $self = shift;
    return $X{$self}{dev} if exists $X{$self}{dev};
}

1;

__END__

=head1 NAME

    BlockDev - library for doing various things on block devices

=head1 SYNOPSIS

    use BlockDev;

    @devices = BlockDev->list();

    $device = BlockDev->new(uuid => "02323-232-22...");
    $device = BlockDev->new(dev => "/dev/sda1");

    ($major, $minor) = $device->rdev;

=head1 Static methods

=over

=item list()

Lists all known block devices (basically listing of F</proc/partitions>)

=item new()

Create a new blockdev object. I<$>

=back

    
