package Nagios::Check::DNS::check_tlsa_record;

use strict;
use warnings;
use feature qw(say switch);
use if $ENV{DEBUG} => 'Smart::Comments';
use Carp;
use Data::Dumper;

#use if $^V >= v5.0.20 => (experimental => gw(smartmatch));
use experimental qw(smartmatch);
use File::Temp;

our $VERSION = '0.1';

#@TODO use only fh of tempfile instead of filename
my $tempfile = File::Temp->new(
    TEMPLATE => '._tlsaXXXX',
    DIR      => '/tmp/',
    SUFFIX   => '.tmp',
);

sub main {
    my $domain   = shift;
    my $port     = shift || 443;
    my $protocol = shift || 'tcp';
    my $validate = validate_tlsa($domain, $port, $protocol);
    return $validate;
}

sub get_tlsa_from_dns {
  #@TODO: multiple TLSA records
    my $domain     = shift;
    my $port       = shift // croak 'Need a port number';
    my $protocol   = shift // 'tcp';
    my $query      = "dig tlsa _$port._$protocol.$domain +short";
    my $dns_return = qx($query);

    if ($dns_return eq '') {
      $dns_return = "No TLSA Record for $domain:$port";
    }

    if ($dns_return =~ /^[_a-z]+/i) {
      $dns_return =~ s/(?<cname>^[_a-z]+.*\.\s+)(?<tlsa>\d+\s+\d+\s+\d+\s+[0-9a-f ]+)/$2/i; 
      $dns_return = uc($dns_return);
    }
    
    # FIXME: what's about the \n? We should cut it!
    return $dns_return;
}

sub get_cert {
    my $domain = shift;
    my $port   = shift;
    my $query;
    my $cert;

    if ($port == 25) {
        $query = "openssl s_client -starttls smtp -connect $domain:$port";

    }
    else {
        $query = "openssl s_client -connect $domain:$port";

    }
    my $same = "< /dev/null 2>/dev/null | openssl x509 -out $tempfile 2>&1";
    $query = "$query $same";

    $cert = qx($query);
    return $cert;
}

sub get_tlsa_from_cert {
    my $cert = shift;
    my $hashit = shift || 'sha256';
    my $gentlsa =
        "openssl x509  -in $tempfile -pubkey | "
      . 'openssl rsa -pubin -inform PEM -outform DER 2>/dev/null | '
      . "openssl $hashit";

    my $tlsa_record = qx($gentlsa) or die "nothing found!\n";
    $tlsa_record =~ s/^.*= (.*$)/$1/gi;
    $tlsa_record = uc($tlsa_record);

    return $tlsa_record;
}

sub check_expiry {
    my $cert         = shift;
    my $check_expiry = "openssl x509 -in $cert -noout -startdate -enddate";
    my $expiry       = qx($check_expiry);

    return $expiry;
}

sub get_tlsa_match_type {
    my $dig_return = shift;
    my $tlsa_usage      = substr($dig_return, 0, 1);
    my $tlsa_selector   = substr($dig_return, 2, 1);
    my $tlsa_match_type = substr($dig_return, 4, 1);
    my $hashit;

    for ($tlsa_match_type) {
        when ('0') { die 'certs will be compared directly' }
        when ('1') { $hashit = 'sha256' }
        when ('2') { $hashit = 'sha512' }
        default { $hashit = 'sha256' }
    }
    return $hashit;
}

sub get_tlsa_dns_record {
    my $dig_return = shift;
    
    my $pattern ;

    my $dig_tlsa = substr($dig_return, 6,);
    $dig_tlsa =~ s/(\S*)\s+(\S*)$/$1$2/;

    return $dig_tlsa;
}

sub get_tlsa_usage {
  # 
  #      @TODO: check certificate trust chain
  #      get_ca_cert()
  #      need to get ca cert for verification if tlsa usage < 2
  #   0: CA Constraints: 
  #         Zertifikat muss von der angegebenen CA stammen.
  #         Der Hash wird aus dem Public Certificate der CA generiert.
  #         x509 Trust chain muss gültig sein.
  #   1: Certificate Constraints: 
  #         Nur das angegebene Zertifikat darf mit der Domain eingesetzt werden. 
  #         Hash wird aus diesem Zertifikat generiert. 
  #         x509 Trust chain muss gültig sein.
  #   2: Trust anchor assertion:  
  #         Das Zertifikat muss von der angegebenen CA stammen.
  #         Hash aus Public Cert der CA. 
  #         Keine Trust chain-Überprüfung.
  #   3: Domain-Issued certificates: 
  #         Nur das angegebene Zertifikat darf mit der Domain eingesetzt werden.
  #         Hash aus dem eigenen Public Cert. 
  #         Keine Trust chain-Überprüfung.
  #
    my $dig_return = shift;
    my $tlsa_usage = substr($dig_return, 0, 1);

    return $tlsa_usage;
}

sub get_tlsa_selector {
  #
  #    0: Gesamtes Zertifikat wird gehashed: 
  #         Record muss mit jeder Zertifikatserneuerung aktualisiert werden.
  #    1: Nur die „SubjectPublicKeyInfo“ wird gehashed: 
  #         Vorteil: Wenn immer derselbe Private Key für die Generierung von
  #         Zertifikaten genutzt wird, muss der TLSA-Record nicht mit jedem 
  #         Zertifikatswechsel erneuert werden.
  #
    my $dns_return = shift;
    my $tlsa_selector = substr($dns_return, 2, 1);

    return $tlsa_selector;
}

# @TODO
# bad-hash.dane.verisignlabs.com -> The TLSA record for this server has an incorrect hash value, although it is correctly signed with DNSSEC
# bad-params.dane.verisignlabs.com -> The TLSA record for this server has a correct hash value, incorrect TLSA parameters, and is correctly signed with DNSSEC. NOTE: The current Firefox plugin accepts these TLSA records as valid.
# bad-sig.dane.verisignlabs.com -> The TLSA record for this server is correct, but the DNSSEC chain-of-trust is broken and/or has a bad signature. NOTE: If you have validation enabled you won't be able to look up the hostname anyway.
# source: http://dane.verisignlabs.com/

sub validate_tlsa {
    my $domain     = shift;
    my $port       = shift;
    my $protocol   = shift;
    my $cert       = get_cert($domain, $port);
    my $dns_return = get_tlsa_from_dns($domain, $port, $protocol);

    if ($cert =~ /.*unable to load certificate.*/) {
        return "WARNING: No SSL-Certificate for $domain:$port";
    }
    if ($dns_return =~ /no tlsa.*$/gi) {
        return "WARNING: $dns_return";
    }

    my $dns_tlsa   = get_tlsa_dns_record($dns_return);
    my $cert_tlsa  = get_tlsa_from_cert($cert);

    if ("$dns_tlsa" ne "$cert_tlsa") {
        return "CRITICAL: TLSA Record for $domain:$port is not valid";
    }

    return "OK: TLSA Record for $domain:$port is valid";
}

# vim: ft=perl ts=2 sw=2 foldmethod=indent
1;
