package Nagios::Check::DNS::check_tlsa_record;

use strict;
use warnings;
use feature qw(say switch);
use Carp;
use Data::Dumper;
use if $ENV{DEBUG} => 'Smart::Comments';

#use if $^V >= v5.0.20 => (experimental => gw(smartmatch));
use experimental qw(smartmatch);
use File::Temp;
use Fcntl qw(F_GETFD F_SETFD FD_CLOEXEC);

our $VERSION = '0.1';

my $dane_pattern = '^(?<record>(?<tlsa_usage>\d+)\s+(?<tlsa_selector>\d+)\s+(?<tlsa_match_type>\d+)\s+(?<tlsa_hash>[0-9a-f ]+))$';
my $with_cname   = '^(?<cname>[_a-z]+.*\n).*';

our $tmpfile = File::Temp->new();
my $fdname = "/dev/fd/' . fileno $tmpfile;
fcntl($tmpfile, F_SETFD, fcntl($tmpfile, F_GETFD, 0) & ~FD_CLOEXEC)
    or die "clear FD_CLOEXEC on $tmpfile: $!\n";

sub main {
    my $domain   = shift;
    my $port     = shift // 443;
    my $protocol = shift // 'tcp';
    my @validate = validate_tlsa($domain, $port, $protocol);
    my $length = @validate;
    my $return = '';

    if ( $length > 1 ) {
      for ( my $i = 0; $i < $length; $i++) {
        $return .= "$validate[$i]\n";
      }
    }
    else {
      $return = $validate[0];
    }
      return $return;
}

sub get_tlsa_from_dns {
    my $domain     = shift;
    my $port       = shift // croak 'Need a port number';
    my $protocol   = shift // 'tcp';
    my $query      = "dig tlsa _$port._$protocol.$domain +short";
    my @dns_return = qx($query);
    my $return_length = @dns_return;
    my $cname;

    for ( my $i = 0; $i < $return_length; $i++)
    {

      if ($dns_return[$i] =~ /^[_a-z]+[a-z0-9]+/i) {
        #$dns_return[$i] = "CNAME: $dns_return[$i]";
        #$dns_return[$i-1] = $dns_return[$i];
        $dns_return[$i] = $dns_return[$i+1];
      }
    }

    # FIXME: what's about the \n? We should cut it!
    return @dns_return;
}

sub get_cert {
    my $domain = shift;
    my $port   = shift;
    my $cmd;
    my $cert;

    if ($port == 25) {
        $cmd = "openssl s_client -starttls smtp -connect $domain:$port";
    }
    else {
        $cmd = "openssl s_client -connect $domain:$port";
    }
    my $same = "< /dev/null 2>/dev/null | openssl x509 -out $fdname 2>&1";
    $cmd .= $same;

    die sprintf "[%s] returned exit:%d signal:%d\n",
        $cmd, $? >> 8, $? & 0xff
        if $?;

    return qx($cmd);
}

sub get_tlsa_from_cert {
    my $cert          = shift;
    my $hashit        = shift // 'sha256';
    my $tlsa_selector = shift // 1;
    my $gentlsa;

    $gentlsa = <<_;
openssl x509  -in $fdname -pubkey |
openssl rsa -pubin -inform PEM -outform DER 2>/dev/null |
openssl $hashit
_


    if ($tlsa_selector == 0) {
      $gentlsa = "openssl x509 -in $fdname -outform DER | openssl $hashit";
    }

    my $tlsa_record = qx($gentlsa) or die "nothing found!\n";
    $tlsa_record =~ s/^.*= (.*$)/$1/gi;
    $tlsa_record = uc($tlsa_record);
    chomp $tlsa_record;

    return $tlsa_record;
}

sub check_expiry {
    my $cert         = shift;
    my $check_expiry = "openssl x509 -in $cert -noout -startdate -enddate";
    my $expiry       = qx($check_expiry);

    return $expiry;
}

sub get_tlsa_match_type {
    my $dig_return = shift;
    my $hashit;
    my $tlsa_match_type;

    if ($dig_return =~ /$dane_pattern/i) {
      $tlsa_match_type = $+{tlsa_match_type};
    }

    if ($tlsa_match_type >= 3) {
       return "Not valid: $tlsa_match_type";
    }

    for ($tlsa_match_type) {
        when ('0') { die 'certs will be compared directly' }
        when ('1') { $hashit = 'sha256' }
        when ('2') { $hashit = 'sha512' }
        default { $hashit = 'sha256' }
    }
    return $hashit;
}

sub get_tlsa_dns_record {
    my $dns_return = shift;
    my $dns_tlsa;

    if ($dns_return =~ /$dane_pattern/i) {
      $dns_tlsa = $+{tlsa_hash};
      $dns_tlsa =~ s/(\S*)\s+(\S*)$/$1$2/;
    }
    return $dns_tlsa;
}

sub get_tlsa_usage {
  #
  #      @TODO: check certificate trust chain
  #      get_ca_cert()
  #      need to get ca cert for verification if tlsa usage < 2
  #   0: CA Constraints:
  #         Zertifikat muss von der angegebenen CA stammen.
  #         Der Hash wird aus dem Public Certificate der CA generiert.
  #         x509 Trust chain muss gültig sein.
  #   1: Certificate Constraints:
  #         Nur das angegebene Zertifikat darf mit der Domain eingesetzt werden.
  #         Hash wird aus diesem Zertifikat generiert.
  #         x509 Trust chain muss gültig sein.
  #   2: Trust anchor assertion:
  #         Das Zertifikat muss von der angegebenen CA stammen.
  #         Hash aus Public Cert der CA.
  #         Keine Trust chain-Überprüfung.
  #   3: Domain-Issued certificates:
  #         Nur das angegebene Zertifikat darf mit der Domain eingesetzt werden.
  #         Hash aus dem eigenen Public Cert.
  #         Keine Trust chain-Überprüfung.
  #
  # https://tools.ietf.org/html/rfc6698#section-2.1.1
  #
    my $dns_return = shift;
    my $tlsa_usage;

    if ($dns_return =~ /$dane_pattern/i) {
      $tlsa_usage = $+{tlsa_usage};
    }
    return $tlsa_usage;
}

sub get_tlsa_selector {
  #
  #    0: Full certificate: the Certificate binary structure as defined
  #          in [RFC5280]
  #    1: SubjectPublicKeyInfo: DER-encoded binary structure as defined
  #          in [RFC5280]
  #         Vorteil: Wenn immer derselbe Private Key für die Generierung von
  #         Zertifikaten genutzt wird, muss der TLSA-Record nicht mit jedem
  #         Zertifikatswechsel erneuert werden.
  #
    my $dns_return = shift;
    my $tlsa_selector;


    if ($dns_return =~ /$dane_pattern/i) {
      $tlsa_selector =  $+{tlsa_selector};
    }

    return $tlsa_selector;

}


sub validate_tlsa {
    my $domain     = shift;
    my $port       = shift;
    my $protocol   = shift;
    my @dns_return = get_tlsa_from_dns($domain, $port, $protocol);
    my $length     = @dns_return;
    my $fail_selector   = 0;
    my $fail_usage      = 0;
    my $fail_match_type = 0;
    my @dns_tlsa;
    my @tlsa_selector;
    my @tlsa_usage;
    my @tlsa_match_type;
    my @return;
    my @cname;

    if ($length == 0) {
      return 'WARNING: No TLSA to check';
    }

    my $cert       = get_cert($domain, $port);

    if ($cert =~ /.*unable to load certificate.*/) {
        return "WARNING: No SSL-Certificate for $domain:$port";
    }
    my $cert_tlsa  = get_tlsa_from_cert($cert);
    chomp $cert_tlsa;

    for (my $i = 0; $i < $length; $i++) {

      if ($dns_return[$i] =~ /no tlsa.*$/gi) {
          return "WARNING: $dns_return[$i]";
      }
      #if ($dns_return[$i] =~ /CNAME: .*$/gi) {
      #  #$dns_return[$i] = $dns_retrun[$i+1];
      #   $i++;
      #}
      if ($dns_return[$i] !~ /CNAME: .*$/gi) {
        $dns_tlsa[$i] = get_tlsa_dns_record($dns_return[$i]);
        $tlsa_selector[$i] = get_tlsa_selector($dns_return[$i]);
        $tlsa_usage[$i] = get_tlsa_usage($dns_return[$i]);
        $tlsa_match_type[$i] = get_tlsa_match_type($dns_return[$i]);

        if ($tlsa_selector[$i] < 0 or $tlsa_selector[$i] > 1) {
          $return[$i] = "CRITICAL: TLSA Selector \'$tlsa_selector[$i]\' for $domain:$port is not valid";
          $fail_selector = 1;
        }

        if ($tlsa_usage[$i] < 0 or $tlsa_usage[$i] > 3) {
          $return[$i] = "CRITICAL: TLSA Usage \'$tlsa_usage[$i]\' for $domain:$port is not valid";
          $fail_usage = 1;
        }

        #if ($tlsa_match_type[$i] !~ /not.*(?<mt>\d+)/i)
        if ($tlsa_match_type[$i] =~ /not.*(?<mt>\d+)/i)
        {
          $return[$i] = "CRITICAL: TLSA Match Type  \'$+{mt}\' for $domain:$port is not valid";
          $fail_match_type = 1;
        }

        if ($fail_match_type != 1) {
          $cert_tlsa  = get_tlsa_from_cert($cert,$tlsa_match_type[$i]);

          if ($fail_selector != 1) {
            $cert_tlsa  = get_tlsa_from_cert($cert,$tlsa_match_type[$i],$tlsa_selector[$i]);
          }

          chomp $cert_tlsa;

        }

        if ($fail_usage != 1 and $fail_selector != 1 and $fail_match_type != 1  ) {
          if ("$dns_tlsa[$i]" ne "$cert_tlsa") {
              $return[$i] = "CRITICAL: TLSA Record for $domain:$port is not valid";
          }
          else {
            $return[$i] =  "OK: TLSA Record for $domain:$port is valid";
          }
        }
      }

      else {
          chomp $dns_return[$i];
          $return[$i] =  "$dns_return[$i] for $domain:$port";
      }
    }

    return @return;
}

# vim: ft=perl ts=2 sw=2 foldmethod=indent
1;
