#! /usr/bin/perl -w

#    Copyright (C) 2011-2013  Christian Arnold
#    Copyright (C) 2013       Matthias Förste
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#    Christian Arnold <arnold@schlittermann.de>, Matthias Förste <foerste@schlittermann.de>

use strict;
use File::Basename;
use Getopt::Long;
use Compress::Zlib;
use Date::Manip;
use Pod::Usage;
use if $ENV{DEBUG} => "Smart::Comments";

my %ERRORS = (
    OK        => 0,
    WARNING   => 1,
    CRITICAL  => 2,
    UNKNOWN   => 3,
    DEPENDENT => 4
);

sub get_status($$$);
sub report($);
sub version($$);

my $ME      = basename $0;
my $VERSION = "2.6";

my %opt = (
    binary => "/usr/bin/rsnapshot",
    maxage => undef,
    date   => "yesterday"
);

MAIN: {
    Getopt::Long::Configure('bundling');
    GetOptions(
        "b|binary=s" => \$opt{binary},
        "s|maxage=i" => \$opt{maxage},
        "d|date=s"   => \$opt{date},
        "h|help" => sub { pod2usage(-verbose => 1, -exitval => $ERRORS{OK}) },
        "m|man" => sub { pod2usage(-verbose => 2, -exitval => $ERRORS{OK}) },
        "V|version" => sub { version($ME, $VERSION); exit $ERRORS{OK}; }
    ) or pod2usage(-verbose => 1, -exitval => $ERRORS{CRITICAL});

    my @logfiles = @ARGV ? @ARGV : glob '/var/log/rsnapshot.log{,.1.gz}';

    my $date = UnixDate(ParseDate($opt{date}), "%d/%b/%Y");
    unless ($date) {
        print "RSNAPSHOT CRITICAL: can't parse date [$opt{date}]\n";
        exit $ERRORS{CRITICAL};
    }

    my $maxage =
      defined $opt{maxage}
      ? time - $opt{maxage}
      : UnixDate(ParseDate($opt{date}), "%s");

    unless (defined $maxage) {
        print
          "RSNAPSHOT CRITICAL: undefined log file max age; this should not happen\n";
        exit $ERRORS{CRITICAL};
    }

    unless (-x $opt{binary}) {
        print
          "RSNAPSHOT CRITICAL: binary '$opt{binary}' - not found or not executable\n";
        exit $ERRORS{CRITICAL};
    }

    my $status = get_status([@logfiles], $date, $maxage);
    report($status);
}

sub get_status($$$) {

    my ($logfiles, $date, $maxage) = @_;
    my ($error, $found_in_file);

    for my $logfile (@{$logfiles}) {

        unless (-e $logfile) {
            print "RSNAPSHOT CRITICAL: logfile '$logfile' - don't exists\n";
            exit $ERRORS{CRITICAL};
        }

        if (-z $logfile) {
            print "RSNAPSHOT WARNING: logfile $logfile - has zero size\n";
            exit $ERRORS{WARNING};
        }

        next if (stat $logfile)[9] < $maxage;

        if ($logfile =~ /\.gz$/) {
            my $gz = gzopen($logfile, "rb")
              or print "RSNAPSHOT CRITICAL: Cannot open $logfile: $gzerrno\n"
              and exit $ERRORS{CRITICAL};
            while ($gz->gzreadline($_) > 0) {
                next unless (/^\[$date/);
                $found_in_file = $logfile;
                if (/^\[$date:.*ERROR/) {
                    $error = $_;
                    last;
                }
            }
            print "RSNAPSHOT CRITICAL: Error reading from $logfile: $gzerrno\n"
              and exit $ERRORS{CRITICAL}
              if $gzerrno != Z_STREAM_END;
            $gz->gzclose();
        } else {
            open(FH, $logfile)
              or print "RSNAPSHOT CRITICAL: $logfile - $!\n"
              and exit $ERRORS{CRITICAL};
            while (<FH>) {
                $found_in_file = $logfile;
                next unless (/^\[$date/);
                if (/^\[$date:.*ERROR/) {
                    $error = $_;
                    last;
                }
            }
            close(FH);
        }

        last if $found_in_file;

    }

    $error |= "can't find rsnapshot run from [$date]\n" unless ($found_in_file); 

    if ($error) {
        print "RSNAPSHOT CRITICAL: $error";
        exit $ERRORS{CRITICAL};
    } else {
        print "RSNAPSHOT OK: no errors in $found_in_file\n";
        exit $ERRORS{OK};
    }
}

sub version($$) {
    my $progname = shift;
    my $version  = shift;

    print <<_VERSION;
$progname version $version
Copyright (C) 2011-2013 by Christian Arnold and Schlittermann internet & unix
support.

Copyright (C) 2013 by Matthias Förste and Schlittermann internet & unix support.

$ME comes with ABSOLUTELY NO WARRANTY. This is free software,
and you are welcome to redistribute it under certain conditions.
See the GNU General Public Licence for details.
_VERSION
}

=head1 NAME

check_rsnapshot - nagios plugin to check for errors on rsnapshot logfile

=head1 SYNOPSIS

check_release [-b|--binary string]
              [-s|--maxage integer]
              [-d|--date string]
              [-h|--help]
              [-m|--man]
              [-V|--version]
              [logfile1 logfile2 ..]

=head1 OPTIONS

=over

=item B<-b>|B<--binary> I<string>

rsnapshot binary (default: /usr/bin/rsnapshot)

=item B<-s>|B<--maxage> I<integer>

Files modified more than B<--maxage> seconds ago will be silently ignored.
(default: same as B<--date>)

=item B<-d>|B<--date> I<string>

Parse date for rsnapshot logfile. (default: yesterday)

=item B<-h>|B<--help>

Print detailed help screen.

=item B<-m>|B<--man>

Print manual page.

=item B<-V>|B<--version>

Print version information.

=back

=head1 DESCRIPTION

This plugin checks rsnapshot logfile for errors.

=head11 EXAMPLES

check_rsnapshot -s 172800 -l /var/log/rsnapshot/rsnapshot.log -d "2 days ago" 

=head1 VERSION

This man page is current for version 2.6 of check_rsnapshot.

=head1 AUTHORS

2011-2013 Originally written by Christian Arnold L<arnold@schlittermann.de>.
2013      Fixes by Matthias Förste L<foerste@schlittermann.de>.

=head1 COPYRIGHT

Copyright (C) 2011-2013 by Christian Arnold and Schlittermann internet & unix
support.  Copyright (C) 2013 by Matthias Förste and Schlittermann internet &
unix support.  This is free software, and you are welcome to redistribute it
under certain conditions.  See the GNU General Public Licence for details.

=cut
