#! perl
use 5.014;
use strict;
use warnings;
use Test::More;
use File::Temp;
use Test::Exception;
use Storable qw(freeze);

my $tmp = File::Temp->new;
$tmp->print(<<__);
# kommentar
    a # more comment
    b
    c ns1 ns2 # comment
    d
__
$tmp->flush;


sub dig_serial { (split " ", `dig +short SOA @_`)[2] }
sub dig_ns { sort map { /(\S+?)\.?$/ } `dig +short NS @_` }

# we require it, it's not a normal module
use blib;
use_ok('Nagios::Check::DNS::delegation')
  or BAIL_OUT q{can't require the module};


subtest 'tools' => sub {
    is_deeply [sort +uniq(qw(a b a c))], [qw(a b c)] => 'uniq helper';

    # get_domains should read a list of names, either from a file
    # or from the arguments, or from a combination of both
    is_deeply [get_domains(sources => [qw/a b c/])], [qw(a b c)] => 'domains from list';
    is_deeply [get_domains(sources => ["$tmp"])],    [qw(a b c d)] => 'domains from file';
    is_deeply [get_domains(sources => ['a', "$tmp", 'z'])], [qw(a a b c d z)] => 'domains from args and file';

    is_deeply {read_override("$tmp")}, { c => ['ns1', 'ns2'] }, 'override file';

    my %google = ( nameservers => [qw/8.8.8.8 8.8.4.4/]);
    my %level3 = ( nameservers => [qw/209.244.0.3 209.244.0.4/]);

    my $r1a = Net::DNS::Resolver->new(%google);
    my $r1b = Net::DNS::Resolver->new(%google);

    my $r2a = Net::DNS::Resolver->new(%level3);
    my $r2b = Net::DNS::Resolver->new(%level3);

    is $r1a, $r1b => 'same google';
    is $r2a, $r2b => 'same level3';
    isnt $r1a, $r2a => 'not same google/level3';

    my (@a, @b);
    @a = qw[8.8.8.1];
    my $r3a = Net::DNS::Resolver->new(nameservers => \@a);
    @a = qw[8.8.4.2];
    my $r3b = Net::DNS::Resolver->new(nameservers => \@a);
    isnt $r3a, $r3b => 'same ref, but not same object';

    @b = @a;
    is freeze(\@a), freeze(\@b) => 'frozen lists';
    my $r3c = Net::DNS::Resolver->new(nameservers => \@b);
    is $r3b, $r3c => 'same servers, but not same ref';

};



for (qw(heise.de schlittermann.de google.com debian.org example.org)) {

    subtest $_ => sub {

        throws_ok { ns($_, nameservers => [qw/8.8.8.8/], aa => 1) }
        qr/no aa/ => 'not authoritive @8.8.8.8';
        is_deeply [ns($_)], [dig_ns($_)] => "ns \@default";
        is_deeply [ns($_, nameservers => [qw/8.8.4.4/])],
          [dig_ns('@8.8.4.4', $_)] => "ns \@8.8.4.4";
        is serial($_, nameservers => [qw/8.8.8.8/]),
          dig_serial('@8.8.8.8', $_) => 'serial';
    };

}

# ns for some domain we're not the master for, should be refused
throws_ok { ns('example.org', nameservers => [qw/f.nic.de a.nic.de/]) }
qr/^REFUSED/ => 'throws on refused query';
throws_ok { ns('safasdfasdfrandomadsfefvddeas') }
qr/^NXDOMAIN/ => 'throws on nx domain';

ok ns_ok('schlittermann.de', ['212.80.235.130']) => 'ns_ok for schlittermann.de';
throws_ok { ns_ok('heise.de', ['212.80.235.130']) } qr/no aa|differ|REFUSED/ => 'ns_ok for heise.de';
throws_ok { ns_ok('heise.de', ['8.8.8.8']) } qr/no aa|differ|REFUSED/ => 'ns_ok for heise.de';

# serial

done_testing();

# vim:sts=4 sw=4 ts=8:
