package Joker;

use Carp;
use Moose;
use MooseX::SemiAffordanceAccessor;
use LWP::UserAgent;
use Joker::Result;
use if $ENV{DEBUG} => 'Smart::Comments';

has [qw(username password)] => (
    isa      => 'Str',
    is       => 'ro',
    required => 1
);

has uri => (
    isa     => 'Str',
    is      => 'ro',
    default => 'https://dmapi.ote.joker.com/request/',    # default test api
    initializer => sub {    # append the trailing slash if missing
        my ($self, $value, $writer) = @_;
        return if $value =~ m{/$};
        $writer->("$value/");
    },
);

has id_file => (
    isa     => 'Str',
    is      => 'ro',
    default => $ENV{JOKER_ID_FILE} // "$ENV{HOME}/.joker-auth-id",
);

has ua => (
    isa      => 'LWP::UserAgent',
    is       => 'ro',
    init_arg => undef,
    default  => sub { LWP::UserAgent->new },
);

has _auth_id => (
    isa      => 'Str',
    is       => 'ro',
    init_arg => undef,
    lazy     => 1,
    builder  => '_build_auth_id',
);

has last_response => (
    is       => 'ro',
    isa      => 'Joker::Response',
    init_arg => undef,
    writer   => '_set_last_response',
);

sub _init_uri {
    die "@_";
}

sub request {
    my ($self, $type) = (shift, shift);
    my %parm = (
        'Auth-Sid' => $self->_auth_id,
        @_,
    );
    my $req = HTTP::Request->new(
        GET => $self->uri . "$type?" . join '&',
        map { "$_=$parm{$_}" } keys %parm
    );
    my $result = $self->ua->request($req);
    ### $result
    croak $result->status_line if not $result->is_success;
    return Joker::Result->new(response => $result->content);
}

sub _login {
    my $self = shift;
    my $req =
      HTTP::Request->new(GET => $self->uri
          . 'login?'
          . 'username='
          . $self->username . '&'
          . 'password='
          . $self->password);
    my $result = $self->ua->request($req);

    croak $result->status_line if not $result->is_success;
    return $1 if $result->content =~ /^Auth-Sid:\s+(\S+)/m;

    croak q{Can't get Auth-Sid};
}

sub _build_auth_id {
    my $self = shift;
    open(my $f, '<', "$ENV{HOME}/.joker.auth-id") or return '';
    chomp(my $_ = <$f>);
    return $_ // '';
}

__PACKAGE__->meta->make_immutable;

__END__

=head1 NAME

Joker - simple OO interface to the Joker domain registry

=head1 SYNOPSIS

 use Joker;

=head1 DESCRIPTION

=head1 CONSTRUCTOR

The B<new> method serves as a constructor. It requires a hash with the
following attributes:

=over

=item B<username> = I<username>

The username, that is your Joker.com account.

=item B<password> = I<password>

The password for the account above.

=back

Optional attributes are:

=over

=item B<ur> = I<uri of the DMAP server>

The URI where the requests are sent to. (default:
https://dmapi.ote.joker.com/) For production use remove the "ote" from
the URI.

=item B<id_file> = I<id file>

The temporary store for the joker auth-id. The first login returns a
joker auth-id (cookie). This cookie may be reused on further requests
until it expires. (default: C<$ENV{JOKER_AUTH_ID_FILE}> //
C<$ENV{HOME}/.joker-auth-id>)

=back

=head1 METHODS

=head2 B<request>(I<request>, I<request params>)

Send a request to the DMAPI. Returns a L<Joker::Result> object. Please
refer to the L<DMAPI specs|https://dmapi.joker.com/docs/DMAPI-ext.txt>.

=cut
