#! /usr/bin/perl

use strict;
use warnings;
use Test::More qw(no_plan);
use File::Temp;

sub mydie(@);
sub mk_tmp_zones($@);

# is the module usable?
BEGIN {
    use_ok "DNStools::UpdateSerial";
}

# does it have all the functions?
can_ok("DNStools::UpdateSerial" => qw(uniq zones changed_zones update_index
    signature_expired need_rollover done_rollover begin_rollover end_rollover
    unlink_unused_keys include_keys sign update_serial mk_zone_conf file_entry
    server_reload dnssec_enabled));

# uniq
ok(@{[sort (uniq(qw(a a b)))]} ~~ @{[qw(a b)]}, 'uniq');

# changed zones
{
    my $m = File::Temp::tempdir(CLEANUP => 1) or mydie "Can't create master dir";
    $config{master_dir} = $m;
    my @z = qw(nostamp.tld);
    mk_tmp_zones($m, @z) or mydie "Can't make zones";
    ok(@{[changed_zones]} ~~ @{[@z]}, "'changed_zones' detects missing .stamp");
}

{
    my $m = File::Temp::tempdir(CLEANUP => 1) or mydie "Can't create master dir";
    $config{master_dir} = $m;
    my @z = qw(newstamp.tld);
    mk_tmp_zones($m, @z) or mydie "Can't make zones";
    my $sf = "$m/$z[0]/.stamp";
    open F, ">$sf" or mydie "Can't create stamp '$sf': $!";
    ok(@{[changed_zones]} ~~ @{[()]}, "'changed_zones' skips zones unless .stamp is older");
}

{
    my $m = File::Temp::tempdir(CLEANUP => 1) or mydie "Can't create master dir";
    $config{master_dir} = $m;
    my @z = qw(newzone.tld);
    mk_tmp_zones($m, @z) or mydie "Can't make zones";
    my $sf = "$m/$z[0]/.stamp";
    open F, ">$sf" or mydie "Can't create stamp file '$sf': $!";
    my $zf = "$m/$z[0]/$z[0]";
    my $atime = (stat "$zf")[8] or mydie "Can't stat zone file '$zf': $!";
    my $utime = time;
    utime $atime, $utime - 1, $sf or mydie "Can't utime .stamp file '$sf': $!";
    utime $atime, $utime, $zf or mydie "Can't utime zone file '$zf': $!";
    ok(@{[changed_zones]} ~~ @{[@z]}, "'changed_zones' detects zones with older .stamp");
}

# update_index
{
    eval { update_index 'foo' };
    ok($@, "'update_index' dies on missing index zone file");
}

## signature_expired
#$m = File::Temp::tempdir(CLEANUP => 1) or mydie "Can't create master dir";
#@z = qw(sigtest.tld);
#mk_tmp_zones($m, @z) or mydie "Can't make zones";
#

# maybe skip some tests
SKIP: {

    skip 'idn broken or not installed?', 2 unless eval {
        local $SIG{__WARN__} = sub {};
        chomp(my $r = qx(idn --quiet sub.tld));
        $r eq 'sub.tld'
    };

    # zones
    eval { zones 'foo' };
    ok($@, "'zones' dies on missing zone file");

    my @z = qw(föö.bär föö.bäz);
    my @zi = qw(xn--f-1gaa.xn--br-via xn--f-1gaa.xn--bz-via);
    my $m = File::Temp::tempdir(CLEANUP => 1) or mydie "Can't create master dir";
    mk_tmp_zones($m, @zi) or mydie "Can't make zones";
    $config{master_dir} = $m;
    ok(@{[zones @z]} ~~ @{[@zi]}, "'zones' internationalizes the zone name");

}

sub mydie(@) { 
    my ($p, $f, $l) = caller;
    die "\n\n", @_, " at $f line $l in package $p\n\n";
}
# we expect the master dir and the internationalized domain names as argument
sub mk_tmp_zones($@) {

    my ($m, @z) = @_;
    return unless $m and @z;

    for (@z) {
        mkdir "$m/$_" or mydie "Can't create zone dir";
        open F, ">$m/$_/$_" or mydie "Can't create zone file";
        close F;
    }

    return 1;

}

__END__

eval { get_config("xxx|xxx", "yyy|yyy") };
ok($@, "dies on missing config");

my ($tmp, %cf);

# prepare some simple sample config
$tmp = File::Temp->new();
print {$tmp} <<__EOF;
# comment
abc = xyz
other =    value with space
__EOF
close($tmp);

# the files is specified, it should find the first 
# existing
%cf = get_config("xxx|xxx", $tmp->filename);
ok(%cf, "got config");
is($cf{abc} => "xyz", "simple value");
is($cf{other} => "valuewithspace", "spaced value");

# it should find the file specified in $ENV{DNSTOOLS_CONF}
$ENV{DNSTOOLS_CONF} = $tmp->filename;
%cf = ();
%cf = get_config("xxx|xxx", $tmp->filename);
ok(%cf, "got config from \$DNSTOOLS_CONF");
is($cf{abc} => "xyz", "simple value");
is($cf{other} => "valuewithspace", "spaced value");
