
#    Copyright (C) 2011 Matthias Förste
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#    Matthias Förste <foerste@schlittermann.de>

=encoding utf8
=cut

package Ius::Dav::Htpasswd;

use strict;
use warnings;

use Apache::Htpasswd qw();
use AppConfig qw();
use File::Path qw(rmtree);
use POSIX qw();
use String::MkPasswd qw();

BEGIN {

    our ( $VERSION, @ISA, @EXPORT_OK );
    use Exporter;

    # set the version for version checking
    $VERSION = 0.2;

    @ISA       = qw(Exporter);
    @EXPORT_OK = qw(readconfig mkpasswd useradd userdel userexpiry usage);
}

sub usage {

    use Pod::Usage;
    use Pod::Find qw(pod_where);

    pod2usage( -input => pod_where( { -inc => 1 }, __PACKAGE__ ), @_ );

}

sub readconfig {

    my $conf = new AppConfig(
        qw(
          expiry=i
          expiry_min=i
          expiry_max=i
          dav_base_local=s
          dav_base_remote=s
          htpasswd=s
          conf_d=s
          www_user=s
          www_group=s
          master_user=s)
    ) or die 'Failed to read config!';
    $conf->file($_)
      for grep -e, map "$_/ius-dav-htpasswd.conf",
      qw(/etc/ius-dav-htpasswd /usr/local/etc/ius-dav-htpasswd ~/.ius-dav-htpasswd .);
    return { $conf->varlist('.') };

}

sub validate {

    my ( $conf, $user, $expiry ) = @_;

    return unless $user =~ /^[[:alnum:]_]+$/;

    if ( defined $expiry ) {
        return unless $expiry =~ /^[0-9]+$/;
        return
          unless $expiry >= $conf->{expiry_min}
              and $expiry <= $conf->{expiry_max};
    }

    return 1;

}

sub useradd {

    my ( $conf, $user, $pass, $expiry ) = @_;

    for (
        qw(expiry expiry_min expiry_max dav_base_local htpasswd conf_d www_user www_group)
      )
    {
        die "Can't determine '$_' - please check configuration"
          unless defined $conf->{$_};
    }

    $expiry = $conf->{expiry} unless defined $expiry and $expiry ne '';
    die 'Invalid input' unless validate $conf, $user, $expiry;

    my $user_dir = "$conf->{dav_base_local}/$user";
    mkdir "$user_dir" or die "Can't mkdir '$user_dir': $!";

    my ( $www_user, $www_group ) = @{$conf}{qw(www_user www_group)};
    my $www_uid = getpwnam $www_user  or die "Can't getpwnam '$www_user'";
    my $www_gid = getgrnam $www_group or die "Can't getgrnam '$www_group'";
    chown $www_uid, $www_gid, "$user_dir"
      or die "Can't chown, '$www_uid', '$www_gid', '$user_dir': $!";

    my $htpasswd_file = $conf->{htpasswd};
    unless ( -e $htpasswd_file ) {
        open H, '>>', $htpasswd_file or die "Can't create '$htpasswd_file': $!";
        close H;
    }

    my $htpasswd = new Apache::Htpasswd $htpasswd_file;
    $htpasswd->htpasswd( $user, $pass )
      or die $htpasswd->error;
    $htpasswd->writeInfo( $user, time + 24 * 60 * 60 * $expiry )
      or die $htpasswd->error;

    my $master_user = $conf->{master_user};
    my $conf_file   = "$conf->{conf_d}/$user.conf";
    open C, '>', $conf_file or die "Can't open '$conf_file': $!";
    print C <<EOC;
<Directory "$user_dir">
    Dav On
    Order Allow,Deny
    Allow From All
    Deny From None
    AuthType Basic
    AuthName "$user"
    AuthUserFile "$htpasswd_file"
    Require user $master_user $user
    # don't allow script execution
    Options Indexes
    AllowOverride None
</Directory>
# vi:ft=apache
EOC
    close C;

    0 == system qw(apache2ctl graceful)
      or die "Can't 'apache2ctl graceful'!";

    return $pass;

}

sub mkpasswd { return String::MkPasswd::mkpasswd; }

sub userdel {

    my ( $conf, $user ) = @_;

    my $rc = 0;

    for (qw(dav_base_local htpasswd conf_d)) {
        die "Can't determine '$_' - please check configuration"
          unless defined $conf->{$_};
    }

    # avoid 'Found = in conditional, should be ==' warnings
    no warnings qw(syntax);
    my $user_dir = "$conf->{dav_base_local}/$user";
    my $err;
    rmtree( $user_dir, error => $err )
      or $rc = -1
      and warn "Error(s) occurred during rmtree '$user_dir': ",
        defined $err ? @{$err} : '';

    my $htpasswd_file = $conf->{htpasswd};
    my $htpasswd      = new Apache::Htpasswd $htpasswd_file;
    $htpasswd->htDelete($user)
      or $rc = -1 and warn "Can't htdelete '$user': ", $htpasswd->error;

    my $conf_file = "$conf->{conf_d}/$user.conf";
    unlink $conf_file
      or $rc = -1 and warn "Can't unlink '$conf_file': $!";

    0 == system qw(apache2ctl graceful)
      or $rc = -1 and warn "Can't 'apache2ctl graceful'!";

    return $rc;

}

sub userexpiry {

    my ($conf) = @_;

    my $rc = 0;

    for (qw(htpasswd)) {
        die "Can't determine '$_' - please check configuration"
          unless defined $conf->{$_};
    }

    my $htpasswd_file = $conf->{htpasswd};
    my $htpasswd      = new Apache::Htpasswd $htpasswd_file;
    # empty @users does not indicate failure
    my @users         = $htpasswd->fetchUsers;
    my $now = time;

    for my $u (@users) {
        if ( my $e = $htpasswd->fetchInfo($u) ) {
            userdel( $conf, $u )
                and warn "Error(s) occured during 'userdel $conf, $u'\n"
              if $now >= $e;
        }
        else {
            warn "Can't get expiry for '$u': ", $htpasswd->error, "\n";
        }
    }

    return $rc;

}

1;

__END__

=pod

=head1 NAME

ius-dav-useradd

ius-dav-useradd.cgi

ius-dav-userdel

ius-dav-userexpiry

Ius::Dav::Htpasswd - Add dav users to htpasswd and remove them after
expiration.

=head1 SYNOPSIS

ius-dav-useradd
   -u|--user user
  [-e|--expiry expiry]

ius-dav-userdel
   -u|--user user

ius-dav-userexpiry

common options
  [-m|--man]
  [-h|--help]

=head1 DESCRIPTION

=head2 ius-dav-useradd

Make a directory for the user. Chown that directory to the webserver user and
group. Add the user to an htpasswd file. Add expiry information to that
htpasswd file. Place a config snippet for the users directory inside a
directory (which is included from the apache config). Reload apache.

=head2 ius-dav-useradd.cgi

This is a CGI Wrapper around ius-dav-useradd.

=head2 ius-dav-userdel

Removes the directory of the user. Removes the user from the htpasswd file.
Removes the config snippet for the users directory. Reload apache.

=head2 ius-dav-userexpiry

Check the htpasswd file and run deletion for any expired users found.

=head1 OPTIONS

=over

=item B<-u|--user> I<user>

The name of the user to add or remove.

=item B<-e|--expiry> I<expiry>

The time in days after which an added user will expire. Defaults to 1.

=back

=head1 FILES

F</etc/ius-dav-htpasswd/ius-dav-htpasswd.conf>

F</usr/local/etc/ius-dav-htpasswd/ius-dav-htpasswd.conf>

F<~/.ius-dav-htpasswd/ius-dav-htpasswd.conf>

F<./ius-dav-htpasswd.conf>

F</srv/dav>

F</etc/apache2/htpasswd>

F</etc/apache2/dav.d>

=head1 REQUIRES

Several perl modules (should be installed automatically). Some kind of cron
daemon to run the user expiry is recommended.

=head1 AUTHOR

Matthias Förste <foerste@schlittermann.de>

=cut

# vim:sts=4 sw=4 aw ai sm:
