package Exim;

=head1 NAME

    Exim - perl interface to the exim binary

=head1 SYNOPSIS

    use Exim;
    my $exim = new Exim;
    
    print $exim->get("spool_directory");

=head1 DESCRIPTION

This package is an interface (wrapper) for calling the Exim binary.

=cut

use strict;
use warnings;
use Carp;

my %basename;
my %binary;
my %config;
my %cmdline;

=head1 METHODS

=head2 Constructor new([options])

Creates a new Exim object. The options are

=over

=item basename => I<basename>

The basename of the Exim binary. (default: exim4)

=item config => I<config>

The name and location of the configuration file. (default:
I<basename>.conf)

=back

=cut

sub new {
    my $class = ref $_[0] ? ref shift : shift;

    bless my $self = \my $anonymous => $class;
    my %arg = @_;

    $basename{$self} = $arg{basename} ? $arg{basename} : "exim4";
    $config{$self}   = $arg{config}   ? $arg{config}   : "$arg{basename}.conf";
    $binary{$self}  = $self->_find_exim;
    $cmdline{$self} = {};

    return $self;
}

=head2 binary(I<>)

Returns the path of the Exim binary.

=cut

sub binary {
    my $self = shift;
    return $binary{$self};
}

=head2 run(I<list>)

Runs Exim and returns the result a C<qx//> does. (Thus depending on
the context as one huge scalar or as a list of scalars.)

=cut

sub run {
    my $self = shift;
    my @cmd  = (
        $binary{$self},
        -C => $config{$self},
        %{ $cmdline{$self} },
        @_
    );

    warn join ":", @cmd, "\n";
    open(EXIM, "-|") or do {
        exec @cmd;
        die "Can't exec: $!\n";
    };

    my @rc = <EXIM>;

    close(EXIM);
    return @rc if wantarray;
    return $rc[0];

}

=head2 get_option(I<name>)

Returns the value of the named configuration parameter/option.
(The may be passed with or without the leading dollar sign.)

=cut

sub get_option {
    my $self = shift;
    chomp(local $_ = $self->run(-bP => ($_[0] =~ /^\$?(.*)/)));
    croak $_ unless /=/;
    return /[=]\s*(.*)/;    # the [] to get the syntax hiliter right
}

=head2 set_cmdline(I<arg> => I<value>[, ...])

This sets a command line option to be used for all following invocations
of Exim. To remove one option, just use C<undef> as value, to remove all
set command line options, use an emptry list.

=cut

sub set_cmdline {
    my $self = shift;
    

    if (defined $_[1]) { $cmdline{$self}->{ $_[0] } = $_[1] }
    elsif (defined $_[0]) { delete $cmdline{$self}->{ $_[0] } }
    else                  { $cmdline{$self} = {} }

}

=head2 expand(I<string>)

Expands the given string (interface to C<-be>).

=cut

sub expand {
    my $self = shift;
    return $self->run(-be => shift);
}

sub DESTROY {
    my $self = shift;
    delete $basename{$self};
    delete $binary{$self};
    delete $config{$self};
    delete $cmdline{$self};
}

sub _find_exim {
    my $self = shift;
    my %seen;
    foreach (split /:/, "$ENV{PATH}:/usr/local/sbin:/usr/sbin:/sbin") {
        next if $seen{$_};
        return "$_/$basename{$self}" if -x "$_/$basename{$self}";
    }
    croak "Can't find the exim binary ($basename{$self})\n";
}

1;

=head1 AUTHOR

Heiko Schlittermann L<<hs@schlittermann.de>>

=cut

__END__
