#! perl

use 5.010;
use Test::More;
use strict;
use warnings;
use File::Basename;

use_ok 'DNS::Vi' or BAIL_OUT 'DNS::Vi not found!';
can_ok 'DNS::Vi', qw(ttl2h h2ttl parse nice delta edit update show);

# TODO: more tests!
is ttl2h(86400), '1d',  '-> 1d';
is h2ttl('1d'),  86400, '<- 1d';

sub slurp {
    local $/ = undef;
    open(my $x, '<', shift);
    return <$x>;
}

# the expect must match the t/samples/? files
my %expect = (
    axfr => {
        RRSETS => 64,
        SOA    => 1,
        NS     => 2,
        A      => 6,
        TXT    => 3,
        AAAA   => 1,
        MX     => 5,
    },
    a => {
        RRSETS => 32,
        SOA    => 1,
        NS     => 16,
        A      => 9,
        TXT    => 4,
        AAAA   => 0,
        MX     => 0,
	SOA    => {
	    'km21.de.' => ['pu.km21.de. hostmaster.schlittermann.de. 1397766411 86400 7200 604800 300'],
	},
    },
    b => {
        RRSETS => 11,
        SOA    => {
	    'km21.de.' => ['pu.schlittermann.de. hostmaster.schlittermann.de. 1397766411 86400 7200 604800 300'],
	},
        NS     => 2,
        A      => 1,
        TXT    => 3,
        AAAA   => 0,
        MX     => 0,
	CNAME  => {
	    'foo.km21.de.' => [qw/tte.km21.de./],
	    'bar.km21.de.' => [qw/tte.km21.de./],
	    'baz.km21.de.' => [qw/tte.km21.de./],
	    'faz.km21.de.' => [qw/a.b.c./],
	},
    },
    c => {
        RRSETS => 12,
        SOA    => 1,
        NS     => 2,
        A      => 2,
        TXT    => 5,
        AAAA   => 0,
        MX     => 2,
    },
    d => {
        RRSETS => 15,
        SOA    => 1,
        NS     => 2,
        A      => [sort qw/
	    abc.de.km21.de.
	    foo.km21.de.
	    tte.km21.de.
	    tte.km21.de.
	    bar.km21.de.
	/],
        TXT    => 5,
        AAAA   => 0,
        MX     => 2,
    },
    'frey-1' => {
	RRSETS => 6,
	SOA => { 'example.com.' => ['p.example.com. hostmaster.example.com. 47 3600 900 604800 300'] },
	A => { 'p.example.com.' => ['1.1.1.1', '2.2.2.2'] },
	NS => { 'example.com.' => ['p.example.com.'] },
	CNAME => {
	    'proxy.mm.frey.example.com.' => ['uxa.frey.example.com.'],
	    'portal.mm.frey.example.com.'=> ['uxb.frey.example.com.'],
	},
    },
);

# uniq list of rrtypes we want to test
my @sets = do {
    my %h;
    @h{ map { keys %{$_} } values %expect } = ();
    grep { $_ ne 'RRSETS' } sort keys %h;
};

foreach my $sample (sort keys %expect) {
    my $file = "t/samples/$sample";

    subtest "sample $file" => sub {
        my %expect = %{ $expect{$sample} };
        my @zone   = parse(slurp $file);
        is @zone, $expect{RRSETS} => "$sample: $expect{RRSETS} RRSETS";
        foreach my $type (@sets) {
	    if (ref $expect{$type} eq ref[]) {
		my @entries = sort map { $_->{label} } grep { $_->{rrtype} eq $type } map { $_->{rrset} } @zone;
		is_deeply \@entries, $expect{$type} => 'list of labels';
	    }
	    elsif (ref $expect{$type} eq ref{}) {
		foreach my $label (keys %{ $expect{$type} }) {
		    my @entries = sort map { $_->{data} } grep { $_->{label} eq $label and $_->{rrtype} eq $type } map { $_->{rrset} } @zone;
		    is_deeply \@entries, $expect{$type}{$label} => 'complete rrsets',
		}
	    }
	    else {
            is grep({ $_->{rrset}{rrtype} eq $type } @zone),
		$expect{$type}//0 => "number of $type rrsets";
	    }
        }

	my %delta = delta(\@zone, \@zone);
        is @{$delta{add}}, 0 => 'nothing added';
        is @{$delta{del}}, 0 => 'nothing deleted';
      }
}

done_testing();
